#!/bin/bash
# run_fetch_month.sh
# Robust cPanel-safe version (fixed PHP include newline warning):
# - Pass config path via argv to PHP (avoids quoting/newline issues)
# - Creates data_monthly/ if missing
# - Uses script directory as BASE_DIR (portable)

BASE_DIR="$(cd "$(dirname "$0")" && pwd)"
LOG_DIR="${BASE_DIR}/data_monthly"
CFG="${BASE_DIR}/config_fetch_data.php"

export PATH=/usr/local/bin:/usr/bin:/bin
mkdir -p "${LOG_DIR}"

PHP_BIN="/usr/local/bin/php"
PY_BIN="/usr/local/bin/python3"
[ -x "${PHP_BIN}" ] || PHP_BIN="$(command -v php 2>/dev/null)"
[ -x "${PY_BIN}" ] || PY_BIN="$(command -v python3 2>/dev/null)"

echo "[RUN_FETCH_MONTH_SH_START] $(date) user=$(whoami) base_dir=${BASE_DIR}" >> "${LOG_DIR}/cron_fetch_shell.log"

CFG_TMP="${LOG_DIR}/kappa_fetch_cfg.txt"

if [ ! -f "${CFG}" ]; then
  echo "[ERROR] Missing config file: ${CFG}" >> "${LOG_DIR}/cron_fetch_shell.log"
  exit 1
fi
if [ -z "${PHP_BIN}" ] || [ ! -x "${PHP_BIN}" ]; then
  echo "[ERROR] PHP not found." >> "${LOG_DIR}/cron_fetch_shell.log"
  exit 1
fi
if [ -z "${PY_BIN}" ] || [ ! -x "${PY_BIN}" ]; then
  echo "[ERROR] python3 not found." >> "${LOG_DIR}/cron_fetch_shell.log"
  exit 1
fi

# Read config via PHP using argv[1] to avoid newline-in-filename warning.
"${PHP_BIN}" -r '
$cfg = include $argv[1];
echo ($cfg["bar"] ?? "5m")."\n";
echo (int)($cfg["year"] ?? (int)date("Y"))."\n";
echo (int)($cfg["month"] ?? (int)date("n"))."\n";
echo ($cfg["okx_base_url"] ?? "https://www.okx.com")."\n";
echo (!empty($cfg["overwrite_existing"]) ? "1" : "0")."\n";
echo (int)($cfg["max_retries"] ?? 5)."\n";
echo (float)($cfg["sleep_seconds_between_pages"] ?? 0.10)."\n";
echo implode(",", (array)($cfg["coins"] ?? []))."\n";
' "${CFG}" > "${CFG_TMP}" 2>> "${LOG_DIR}/cron_fetch_shell.log"

if [ ! -s "${CFG_TMP}" ]; then
  echo "[ERROR] Failed to extract config into ${CFG_TMP}" >> "${LOG_DIR}/cron_fetch_shell.log"
  exit 1
fi

BAR=$(sed -n '1p' "${CFG_TMP}")
YEAR=$(sed -n '2p' "${CFG_TMP}")
MONTH=$(sed -n '3p' "${CFG_TMP}")
OKX=$(sed -n '4p' "${CFG_TMP}")
OVERWRITE=$(sed -n '5p' "${CFG_TMP}")
RETRIES=$(sed -n '6p' "${CFG_TMP}")
SLEEP=$(sed -n '7p' "${CFG_TMP}")
COINS=$(sed -n '8p' "${CFG_TMP}")

if [ -z "${COINS}" ]; then
  echo "[ERROR] No coins configured in config_fetch_data.php" >> "${LOG_DIR}/cron_fetch_shell.log"
  exit 1
fi

IFS=',' read -ra ARR <<< "${COINS}"

for COIN in "${ARR[@]}"; do
  [ -z "${COIN}" ] && continue
  EXTRA=""
  if [ "${OVERWRITE}" = "1" ]; then EXTRA="--overwrite"; fi

  echo "[FETCH] coin=${COIN} bar=${BAR} year=${YEAR} month=${MONTH}" >> "${LOG_DIR}/cron_fetch_shell.log"

  "${PY_BIN}" "${BASE_DIR}/fetch_month.py" \
    --coin "${COIN}" --bar "${BAR}" --year "${YEAR}" --month "${MONTH}" \
    --base-url "${OKX}" --max-retries "${RETRIES}" --sleep "${SLEEP}" \
    ${EXTRA} >> "${LOG_DIR}/cron_fetch_shell.log" 2>&1
done

echo "[RUN_FETCH_MONTH_SH_END] $(date)" >> "${LOG_DIR}/cron_fetch_shell.log"
