#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os, sys, csv, argparse, traceback
from datetime import datetime, timezone

def utc_iso():
    return datetime.now(timezone.utc).isoformat()

def ensure_dir(p):
    if not os.path.isdir(p):
        os.makedirs(p, exist_ok=True)

def log_line(fp, s):
    line = "[%s] %s" % (utc_iso(), s)
    print(line, flush=True)
    fp.write(line + "\n"); fp.flush()

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--coin", required=True)
    ap.add_argument("--bar", default="5m")
    ap.add_argument("--stitched-file", default=None)
    ap.add_argument("--out-dir", default=None)
    ap.add_argument("--log-dir", default=None)
    ap.add_argument("--fee-pct", type=float, default=0.2)  # example: 0.2%
    args = ap.parse_args()

    base_dir = os.path.dirname(os.path.abspath(__file__))
    stitched_default = os.path.join(base_dir, "data_monthly", "stitched", "%s_%s_all.csv" % (args.coin, args.bar))
    stitched = args.stitched_file or stitched_default
    out_dir = args.out_dir or os.path.join(base_dir, "data_monthly", "sim")
    log_dir = args.log_dir or os.path.join(base_dir, "data_monthly")
    ensure_dir(out_dir); ensure_dir(log_dir)

    log_path = os.path.join(log_dir, "cron_sim.log")
    with open(log_path, "a", encoding="utf-8") as logfp:
        try:
            log_line(logfp, "RUN_START sim coin=%s bar=%s file=%s" % (args.coin, args.bar, stitched))
            if not os.path.exists(stitched):
                log_line(logfp, "MISSING stitched file: %s" % stitched)
                log_line(logfp, "RUN_END sim ok=1 (nothing to do)")
                return 0

            # Dummy sim: buy first candle, sell last candle
            first = None
            last = None
            with open(stitched, "r", encoding="utf-8") as f:
                for line in f:
                    line = line.strip()
                    if not line:
                        continue
                    parts = line.split(",")
                    # ts,o,h,l,c,vol,volCcy,volCcyQuote,confirm
                    ts = int(parts[0])
                    close = float(parts[4])
                    if first is None:
                        first = (ts, close)
                    last = (ts, close)

            if not first or not last:
                log_line(logfp, "NO_DATA")
                log_line(logfp, "RUN_END sim ok=1")
                return 0

            entry_ts, entry_px = first
            exit_ts, exit_px = last
            pnl_pct = (exit_px - entry_px) / entry_px * 100.0
            fee_pct = args.fee_pct
            gross_usd = (entry_px * (pnl_pct / 100.0))
            fee_usd = (fee_pct / 100.0) * entry_px
            net_usd = gross_usd - fee_usd

            out_file = os.path.join(out_dir, "%s_%s_summary.csv" % (args.coin, args.bar))
            with open(out_file, "w", newline="", encoding="utf-8") as out:
                w = csv.writer(out)
                w.writerow(["coin","bar","entry_ts","entry_px","exit_ts","exit_px","pnl_pct","fee_pct","gross_usd","fee_usd","net_usd"])
                w.writerow([args.coin,args.bar,entry_ts,entry_px,exit_ts,exit_px,round(pnl_pct,4),fee_pct,round(gross_usd,6),round(fee_usd,6),round(net_usd,6)])

            log_line(logfp, "SIM_RESULT pnl_pct=%.4f fee_pct=%.4f out=%s" % (pnl_pct, fee_pct, out_file))
            log_line(logfp, "RUN_END sim ok=1")
            return 0

        except Exception as e:
            log_line(logfp, "RUN_END sim ok=0 error=%s" % str(e))
            log_line(logfp, traceback.format_exc())
            return 1

if __name__ == "__main__":
    sys.exit(main())
