#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os, sys, glob, argparse, traceback, time
from datetime import datetime, timezone

def utc_iso():
    return datetime.now(timezone.utc).isoformat()

def ensure_dir(p):
    if not os.path.isdir(p):
        os.makedirs(p, exist_ok=True)

def log_line(fp, s):
    line = "[%s] %s" % (utc_iso(), s)
    print(line, flush=True)
    fp.write(line + "\n"); fp.flush()

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--coin", required=True)
    ap.add_argument("--bar", default="5m")
    ap.add_argument("--data-root", default=None)
    ap.add_argument("--out-dir", default=None)
    ap.add_argument("--log-dir", default=None)
    args = ap.parse_args()

    base_dir = os.path.dirname(os.path.abspath(__file__))
    data_root = args.data_root or os.path.join(base_dir, "daily_data")
    out_dir = args.out_dir or os.path.join(base_dir, "data_monthly", "stitched")
    log_dir = args.log_dir or os.path.join(base_dir, "data_monthly")
    ensure_dir(out_dir); ensure_dir(log_dir)

    log_path = os.path.join(log_dir, "cron_stitch.log")
    with open(log_path, "a", encoding="utf-8") as logfp:
        try:
            inst = args.coin.strip()
            bar = args.bar.strip()
            log_line(logfp, "RUN_START stitch coin=%s bar=%s" % (inst, bar))

            src_dir = os.path.join(data_root, bar, inst)
            files = sorted(glob.glob(os.path.join(src_dir, "*.csv")))
            if not files:
                log_line(logfp, "NO_FILES %s" % src_dir)
                log_line(logfp, "RUN_END stitch ok=1")
                return 0

            out_file = os.path.join(out_dir, "%s_%s_all.csv" % (inst, bar))
            seen = set()
            rows = 0

            with open(out_file, "w", encoding="utf-8") as out:
                for f in files:
                    log_line(logfp, "READ %s" % f)
                    with open(f, "r", encoding="utf-8") as inp:
                        for line in inp:
                            line = line.strip()
                            if not line:
                                continue
                            # dedupe by timestamp (first column)
                            ts = line.split(",", 1)[0]
                            if ts in seen:
                                continue
                            seen.add(ts)
                            out.write(line + "\n")
                            rows += 1

            log_line(logfp, "WROTE %s rows=%d months=%d" % (out_file, rows, len(files)))
            log_line(logfp, "RUN_END stitch ok=1")
            return 0
        except Exception as e:
            log_line(logfp, "RUN_END stitch ok=0 error=%s" % str(e))
            log_line(logfp, traceback.format_exc())
            return 1

if __name__ == "__main__":
    sys.exit(main())
